-- GCI v.5.06® 2025© for LOCK-ON GREECE by =GR= Astr0 via Gemini AI

-- I. ΒΑΣΙΚΕΣ ΠΑΡΑΜΕΤΡΟΙ ΣΥΣΤΗΜΑΤΟΣ
local scan_interval_sec = 15          -- Διάστημα σάρωσης (σε δευτερόλεπτα)
local display_duration_sec = 15       -- Διάρκεια εμφάνισης αναφοράς (σε δευτερόλεπτα)
local cue_range_nm = 5                -- Εμβέλεια για εμφάνιση ρολογιού/σχετικού ύψους (NM)
local min_agl_feet = 3                -- Ελάχιστο ύψος AGL για αναφορά επαφής (πόδια)
local max_reported_contacts = 2       -- Μέγιστος αριθμός επαφών στην αναφορά
local meters_per_nautical_mile = 1852 -- Μετροπή: Μέτρα ανά NM
local gci_enabled = true              -- Ενεργοποίηση/Απενεργοποίηση της λειτουργίας GCI
local player_unit_ref = nil           -- Αναφορά στη μονάδα του παίκτη (δυναμική)
local feet_per_meter = 3.28084        -- Μετατροπή: Πόδια ανά μέτρο
local include_rotary_wing = false     -- Συμπερίληψη ελικοπτέρων
local use_bullseye_mode = false       -- Χρήση αναφοράς BULLSEYE (true/false, default=BRAA)
local los_samples_count = 10          -- Αριθμός δειγμάτων για έλεγχο οπτικής επαφής (LOS)
local los_skip_distance_nm = 3        -- Απόσταση (NM) κάτω από την οποία παρακάμπτεται ο έλεγχος LOS
local sensor_cache_refresh_sec = 30   -- Διάστημα ανανέωσης λίστας φιλικών ραντάρ (sec)
local dead_unit_cleanup_sec = 300     -- Διάστημα καθαρισμού κατεστραμμένων μονάδων (sec)
local report_screen_position = {x = 0.9, y = 0.1} -- Θέση εμφάνισης
local show_column_headers = true      -- Εμφάνιση/Απόκρυψη των επικεφαλίδων των στηλών

-- II. ΠΙΝΑΚΕΣ ΔΕΔΟΜΕΝΩΝ ΑΙΣΘΗΤΗΡΩΝ
local sensorMaximumRanges = {
    ["a-50"] = 350, ["e-3a"] = 320, ["e-2c"] = 200, ["55g6 ewr"] = 215, ["1l13 ewr"] = 160,
    ["ewr p-37 bar lock"] = 180, ["fps-117 dome"] = 250, ["fps-117"] = 250, ["p-19 s-125 sr"] = 140,
    ["p-10 s125 sr"] = 80,    ["dog ear radar"] = 20, ["kub 1s91 str"] = 50, ["FuMG-401"] = 88,
    ["FuSe-65"] = 32, ["Allies_Director"] = 16, ["s-300ps 64h6e sr"] = 180,
    ["s-300 ps 40b6md sr"] = 150, ["patriot str"] = 150, ["hq-7_str_sp"] = 30,
    ["nasams_radar_mpq64f1"] = 40, ["rls_19j6"] = 80, ["rpc_5n62v"] = 40,
    ["snr_75v"] = 53, ["snr s-125 tr"] = 20, ["sa-11 buk sr 9518m1"] = 80,
    ["sa-11 buk ln 9a310m1"] = 50, ["hawk sr"] = 70, ["hawk tr"] = 40,
    ["tor 9a331"] = 25, ["osa 9a33 ln"] = 30, ["roland radar"] = 15,
    ["rapier_fsa_blindfire_radar"] = 10, ["stennis"] = 96, ["kuznecow"] = 96,
    ["vinson"] = 96, ["cvn_71"] = 96, ["cvn_72"] = 96, ["cvn_73"] = 96,
    ["cvn_75"] = 96, ["type_052b"] = 54, ["type_052c"] = 140, ["type_054a"] = 86,
    ["perry"] = 80, ["albatros"] = 16, ["forrestal"] = 96, ["neustrash"] = 15,
    ["lha_tarawa"] = 80, ["MOSCOW"] = 96, ["TICONDEROG"] = 80,    ["PIOTR"] = 135,    
    ["USS_Arleigh_Burke_IIa"] = 80,    ["Type_071"] = 160, ["MOLNIYA"] = 12,
    ["hms_invincible"] = 53, ["leander-gun-achilles"] = 80, ["leander-gun-condell"] = 80,
    ["leander-gun-andromeda"] = 80, ["leander-gun-lynch"] = 96, ["leander-gun-ariadne"] = 80,
    ["zsu-23-4 shilka"] = 8, ["2s6 tunguska"] = 12, ["vulcan"] = 8,
    ["CHAP_PantsirS1"] = 20, ["CHAP_IRISTSLM_STR"] = 135, ["default"] = 25
}

local sensorTypeNames = {
    "A-50", "E-3A", "E-2C", "S-300PS 64H6E sr", "55G6 EWR", "1L13 EWR", "FPS-117 Dome", 
    "p-10 s125 sr", "Patriot str", "HQ-7_STR_SP", "RLS_19J6", "RPC_5N62V", "SNR_75V", 
    "SA-11 Buk LN 9a310M1", "Hawk sr", "Hawk tr", "Tor 9A331", "Osa 9a33 ln", "Roland Radar",
    "kub 1S91 str", "S-300 PS 40b6md sr", "SA-11 Buk SR 9518M1", "ZSU-23-4 Shilka", 
    "2S6 Tunguska", "Vulcan", "FuMG-401", "FuSe-65", "Allies_Director", "HEMTT_C-RAM_Phalanx", 
    "p-19 s-125 sr", "SNR s-125 tr", "rapier_fsa_blindfire_radar", "Gepard", 
    "EWR P-37 BAR LOCK", "Stennis", "Kuznecow", "CVN_75", "Type_052B", "Type_052C", 
    "Forrestal", "LHA-Tarawa", "leander-gun-achilles", "leander-gun-andromeda", 
    "MOSCOW", "TICONDEROG", "PIOTR", "USS_Arleigh_Burke_IIa", "Type_071", "MOLNIYA"
}

-- Τύποι AWACS
local airborneSensorTypes = {
    ["a-50"] = true,
    ["e-3a"] = true,
    ["e-2c"] = true
}

-- Cache και κατάσταση
local sensorCache = {}
local sensorCacheTime = 0
local destroyedUnits = {}
local lastCleanupTime = 0
local validSensorNames = {}
local menuAddedGroups = {} 
for _, t in ipairs(sensorTypeNames) do
    if t and type(t) == "string" then
        validSensorNames[string.lower(t)] = true
    end
end

-- III. ΒΟΗΘΗΤΙΚΕΣ ΣΥΝΑΡΤΗΣΕΙΣ ΛΟΓΙΚΗΣ

-- 1. Παρακολούθηση μονάδων που καταστράφηκαν
world.addEventHandler({
    onEvent = function(event)
        if (event.id == world.event.S_EVENT_KILL or event.id == world.event.S_EVENT_DEAD)
        and event.target then
            local killed = event.target
            if killed.getName and killed:getName() then
                destroyedUnits[killed:getName()] = true
            end
        end
    end
})

-- 2. Υπολογισμός 2D απόστασης
local function calculate2DDistance(unit_a, unit_b)
    local p1 = unit_a:getPosition().p
    local p2 = unit_b:getPosition().p
    return math.sqrt((p1.x - p2.x)^2 + (p1.z - p2.z)^2)
end

-- 3. Λήψη ύψους MSL σε πόδια
local function getAltitudeMSL(unit)
    return unit:getPoint().y * feet_per_meter
end

-- 4. Λήψη ύψους AGL σε πόδια
local function getAGL(unit)
    local pos = unit:getPoint()
    local terrain = land.getHeight({x = pos.x, y = pos.z})
    return (pos.y - terrain) * feet_per_meter
end

-- 5. Έλεγχος οπτικής επαφής (Line of Sight - LOS)
local function checkLOS(sensor, target)
    local sensor_position = sensor:getPoint()
    local target_position = target:getPoint()
    local displacement_x = target_position.x - sensor_position.x
    local displacement_z = target_position.z - sensor_position.z
    local displacement_y = target_position.y - sensor_position.y
    local horizontal_distance = math.sqrt(displacement_x*displacement_x + displacement_z*displacement_z)

    if horizontal_distance == 0 then return true end
    
    local near_threshold_meters = los_skip_distance_nm * meters_per_nautical_mile
    if horizontal_distance < near_threshold_meters then return true end
    
    -- Δειγματοληψία
    local sample_count = math.min(los_samples_count, math.max(4, math.floor(horizontal_distance / 500)))
    
    local current_sample_index = 1
    local interpolation_step = 1 / sample_count
    while current_sample_index < sample_count do
        local factor_t = current_sample_index * interpolation_step
        
        -- Υπολογισμός σημείου ελέγχου (x, z)
        local current_x = sensor_position.x + displacement_x * factor_t
        local current_z = sensor_position.z + displacement_z * factor_t
        
        -- Υπολογισμός του ύψους της ευθείας στο σημείο αυτό
        local los_height = sensor_position.y + displacement_y * factor_t
        
        -- Λήψη του ύψους του εδάφους
        local current_terrain = land.getHeight({x = current_x, y = current_z})
        
        -- Έλεγχος αν το έδαφος είναι πάνω από την ευθεία LOS
        if current_terrain > los_height + 0.5 then 
            return false -- Η οπτική επαφή έχει διακοπεί
        end
        current_sample_index = current_sample_index + 1
    end
    
    return true -- Οπτική επαφή διατηρείται
end

-- 6. Υπολογισμός bearing ως προς τον εχθρό
local function calculateBearing(origin, destination)
    local o_pos = origin:getPoint()
    local d_pos = destination:getPoint()
    local angle_rad = math.atan2(d_pos.z - o_pos.z, d_pos.x - o_pos.x)
    local angle_deg = math.deg(angle_rad)
    if angle_deg < 0 then angle_deg = angle_deg + 360 end
    return angle_deg
end

-- 7. Υπολογισμός Aspect (Hot, Cold, Flanking)
local function determineAspect(player_unit, target_unit)
    local p_point = player_unit:getPoint()
    local t_point = target_unit:getPoint() 
    
    local dx = p_point.x - t_point.x
    local dz = p_point.z - t_point.z
    
    local angle_to_player_rad = math.atan2(dz, dx)
    local angle_to_player_deg = math.deg(angle_to_player_rad)
    if angle_to_player_deg < 0 then angle_to_player_deg = angle_to_player_deg + 360 end

    local heading_rad = nil
    if target_unit.getPosition and target_unit:getPosition() and target_unit:getPosition().heading then
        heading_rad = target_unit:getPosition().heading
    else
        local velocity = target_unit:getVelocity()
        if velocity then heading_rad = math.atan2(velocity.z, velocity.x) end
    end

    if not heading_rad then return "Unknown" end
    
    local target_heading_deg = math.deg(heading_rad)
    if target_heading_deg < 0 then target_heading_deg = target_heading_deg + 360 end
    
    local relative_angle = math.abs(angle_to_player_deg - target_heading_deg)
    if relative_angle > 180 then relative_angle = 360 - relative_angle end

    if relative_angle < 45 then return "hot" elseif relative_angle > 135 then return "cold" else return "flanking" end
end

-- 8. Υπολογισμός θέσης Ο'clock
local function getCockpitClockCue(player_unit, target_unit)
    local p_pos = player_unit:getPosition()
    local p_heading = p_pos and p_pos.heading
    if p_heading == nil or type(p_heading) ~= "number" then return "Unknown" end
    local heading_deg = math.deg(p_heading)
    if heading_deg < 0 then heading_deg = heading_deg + 360 end
    local bearing = calculateBearing(player_unit, target_unit)
    local relative_bearing = math.floor((bearing - heading_deg + 360) % 360)
    local clock = math.floor((relative_bearing + 15) / 30) % 12
    return (clock == 0 and 12 or clock) .. " o'clock"
end

-- 9. Υπολογισμός σχετικού ύψους (High, Low, Level)
local function getVerticalRelative(player_unit, target_unit)
    local diff = getAltitudeMSL(target_unit) - getAltitudeMSL(player_unit) 
    local alt_diff_threshold = 500
    if math.abs(diff) < alt_diff_threshold then return "level" elseif diff > 0 then return "high" else return "low" end
end

-- 10. Λήψη ονόματος τύπου εναέριας επαφής (με αποκοπή στους 5 χαρακτήρες)
local function getShortTypeName(full_name)
    if not full_name or type(full_name) ~= 'string' then 
        return "BANDITS" 
    end
    -- Αποκοπή των πρώτων 5 χαρακτήρων και μετατροπή σε κεφαλαία
    return string.upper(string.sub(full_name, 1, 5))
end


-- 11. Λήψη του σημείου αναφοράς Bullseye
local function getMissionBullseye(side)
    local bullseye = coalition.getMainRefPoint(side)
    if bullseye and bullseye.x and bullseye.z then
        return {x = bullseye.x, y = bullseye.y or 0, z = bullseye.z}
    end
    return nil
end

-- 12. Υπολογισμός bearing και distance από το Bullseye
local function calculateBullseyeRef(unit, side)
    local bullseye = getMissionBullseye(side)
    if not bullseye then return nil, nil end
    local unitPos = unit:getPoint()
    local angle_rad = math.atan2(unitPos.z - bullseye.z, unitPos.x - bullseye.x)
    local angle_deg = math.deg(angle_rad)
    if angle_deg < 0 then angle_deg = angle_deg + 360 end
    local bearing = math.floor(angle_deg + 0.5)
    local dx = unitPos.x - bullseye.x
    local dz = unitPos.z - bullseye.z
    local distance_meters = math.sqrt(dx*dx + dz*dz)
    local distance_nm = distance_meters / meters_per_nautical_mile
    return bearing, distance_nm
end

-- 13. Εύρεση μονάδας client
local function locatePlayerAircraft()
    for _, side in ipairs({coalition.side.BLUE, coalition.side.RED}) do
        for _, group in ipairs(coalition.getGroups(side, Group.Category.AIRPLANE)) do
            for _, unit in ipairs(group:getUnits()) do
                if unit and unit:isActive() and unit.getPlayerName and unit:getPlayerName() then
                    return unit, side
                end
            end
        end
        for _, group in ipairs(coalition.getGroups(side, Group.Category.HELICOPTER)) do
            for _, unit in ipairs(group:getUnits()) do
                if unit and unit:isActive() and unit.getPlayerName and unit:getPlayerName() then
                    return unit, side
                end
            end
        end
    end
    return nil, nil
end

-- 14. Λήψη όλων των ενεργών μονάδων
local function fetchAirUnits(side)
    local units = {}
    for _, group in ipairs(coalition.getGroups(side, Group.Category.AIRPLANE)) do
        for _, unit in ipairs(group:getUnits()) do
            if unit and unit:isActive() then table.insert(units, unit) end
        end
    end
    return units
end

local function fetchHeloUnits(side)
    local units = {}
    for _, group in ipairs(coalition.getGroups(side, Group.Category.HELICOPTER)) do
        for _, unit in ipairs(group:getUnits()) do
            if unit and unit:isActive() then table.insert(units, unit) end
        end
    end
    return units
end

-- 15. Λήψη μονάδων radar/AWACS
local function findCoalitionRadars(side)
    local sensors = {}
    local function add(category)
        for _, group in ipairs(coalition.getGroups(side, category)) do
            for _, unit in ipairs(group:getUnits()) do
                if unit and unit:isActive() and unit.getTypeName then
                    local typeName = string.lower(unit:getTypeName())
                    if validSensorNames[typeName] and unit:getLife() > 1 and not destroyedUnits[unit:getName()] then
                        table.insert(sensors, unit)
                    end
                end
            end
        end
    end
    add(Group.Category.AIRPLANE) 
    add(Group.Category.GROUND)   
    add(Group.Category.SHIP) -- ΔΙΟΡΘΩΘΗΚΕ: Λάθος σύνταξη 'Group(category.SHIP)'
    return sensors
end

-- 16. Ενημέρωση και λήψη της cache των ραντάρ
local function getSensorData(side)
    local current_time = timer.getTime()
    if current_time - sensorCacheTime > sensor_cache_refresh_sec then
        sensorCache[side] = findCoalitionRadars(side)
        sensorCacheTime = current_time
    end
    return sensorCache[side] or {}
end

-- 17. Λήψη εμβέλειας αισθητήρα radar
local function fetchSensorRange(unit)
    if not unit or not unit.getTypeName then return sensorMaximumRanges.default end
    local typeName = string.lower(unit:getTypeName())
    return sensorMaximumRanges[typeName] or sensorMaximumRanges.default
end 
 

-- 18. Έλεγχος αν η μονάδα είναι AWACS
local function isAirborneSensor(unit)
    if not unit or not unit.getTypeName then return false end
    local typeName = string.lower(unit:getTypeName())
    return airborneSensorTypes[typeName] or false
end

-- 19. Έλεγχος αν υπάρχει ενεργό AWACS
local function isAWACSActive(side)
    local groups = coalition.getGroups(side, Group.Category.AIRPLANE)
    if not groups then return false end    
    for _, group in ipairs(groups) do
        for _, unit in ipairs(group:getUnits()) do
            if unit and unit:isActive() and unit.getTypeName then
                local typeName = unit:getTypeName()
                local lowerTypeName = string.lower(typeName)
                if airborneSensorTypes[lowerTypeName] and unit:getLife() > 1 then
                    return true
                end
            end
        end
    end
    return false
end

-- 20. Λήψη ενδείξεων O'Clock και σχετικό ύψος
local function getCloseProximityCues(player_unit, target_unit, distance_nm)
    if distance_nm <= cue_range_nm then
        local clockCue = getCockpitClockCue(player_unit, target_unit)
        local altCue = getVerticalRelative(player_unit, target_unit)
        return clockCue, altCue
    end
    return nil, nil
end

-- 21. Επιστρέφει τον τύπο του εχθρού ('Bandits' ή πραγματικό τύπο)
local function identifyTargetType(enemyUnit, awacsActiveStatus)
    if not enemyUnit or not enemyUnit.getTypeName then 
        return "BANDITS" 
    end
    -- Χρήση της νέας συνάρτησης για περιορισμό του ονόματος
    if awacsActiveStatus then 
        return getShortTypeName(enemyUnit:getTypeName()) 
    else 
        return "BANDITS" 
    end
end

-- IV. ΣΥΝΑΡΤΗΣΕΙΣ ΜΟΡΦΟΠΟΙΗΣΗΣ ΚΑΙ ΚΥΡΙΑ ΛΟΓΙΚΗ

-- 22. Μορφοποίηση απόστασης (με προπορευόμενα μηδενικά)
local function formatNMString(distance_nm)
    return string.format("%03dNM", math.floor(distance_nm + 0.5))
end

-- 23. Λογική ύψους (Μετατροπή σε κεφαλαία 'FT' / 'FL')
local function formatFLorFeet(alt_value) 
    local fl_threshold_feet = 10000 
    local s
    if alt_value >= fl_threshold_feet then 
        s = string.format("FL%03d", math.floor(alt_value / 100)) 
    else
        local rounded_alt = math.floor(alt_value / 100) * 100
        s = string.format("%dFT", rounded_alt) 
    end
    
    -- Εφαρμογή της ζητούμενης τροποποίησης: προσθήκη κενού μπροστά αν το μήκος < 6
    if string.len(s) < 6 then
        s = "   " .. s
    end
    
    return s
end

-- 24. ΟΡΙΣΜΟΣ ΣΤΑΘΕΡΩΝ ΠΛΑΤΩΝ ΚΑΙ ΔΙΑΧΩΡΙΣΤΩΝ (ΧΡΗΣΙΜΟΠΟΙΟΥΝΤΑΙ ΓΙΑ ΤΑ ΔΕΔΟΜΕΝΑ)
-- Αυτά τα πλάτη και οι διαχωριστές ΔΕΝ πρέπει να αλλάξουν για να διατηρηθεί η στοίχιση των δεδομένων.
local W_STATIC = {
    W_TYPE = 20, W_BULLSEYE_REF = 12, W_BRG = 5, W_DIST = 6, W_ALT = 7, W_ASPECT = 8
}
local S_STATIC = {
    S_TYPE_TO_NEXT = "    ", S_DIST_TO_ALT = "   ", S_BRG_TO_DIST = "  ", S_ALT_TO_ASPECT = "     " 
}


-- 25. Δημιουργία της γραμμής αναφοράς επαφής
local function generateReportLine(type_name_raw, bearing_val, dist_nm, alt_unit, aspect_val, clock_cue, alt_cue, bullseye_brg, bullseye_dist, isBullseye, awacsActiveStatus)    
    
    local W, S = W_STATIC, S_STATIC -- Χρήση σταθερών πλατών/διαχωριστών
    
    -- 1. Format Type (Right-Aligned in W_TYPE)
    local type_uppercase = string.upper(type_name_raw)
    local type_aligned = string.format("%"..W.W_TYPE.."s", type_uppercase)
    
    local line = type_aligned .. S.S_TYPE_TO_NEXT

    local altitude_formatted = formatFLorFeet(getAltitudeMSL(alt_unit))
    local aspect_uppercase = string.upper(aspect_val)

    -- 2. Format ALT and ASPECT (Common to both modes)
    local alt_aligned = string.format("%"..W.W_ALT.."s", altitude_formatted)
    -- ΑΡΙΣΤΕΡΗ ΣΤΟΙΧΙΣΗ ΓΙΑ ΤΟ ASPECT
    local aspect_aligned = string.format("%-"..W.W_ASPECT.."s", aspect_uppercase)
    
    -- 3. Format Cue String
    local cue_string = ""
    if clock_cue and alt_cue then
        cue_string = string.format(" (%s, %s)", string.upper(clock_cue), string.upper(alt_cue))
    end
    
    if isBullseye and bullseye_brg and bullseye_dist then
        -- BULLSEYE: BRG / DIST
        local bullseye_dist_formatted = formatNMString(bullseye_dist)
        local bullseye_ref_str_value = string.format("%03.0f° / %s", bullseye_brg, bullseye_dist_formatted)
        local bullseye_ref_aligned = string.format("%"..W.W_BULLSEYE_REF.."s", bullseye_ref_str_value)

        line = line .. bullseye_ref_aligned .. S.S_DIST_TO_ALT ..
               alt_aligned .. S.S_ALT_TO_ASPECT ..
               aspect_aligned .. cue_string
    else
        -- BRAA: BRG | DIST
        local bearing_formatted_string = string.format("%03.0f°", bearing_val or 0) 
        local dist_aligned = string.format("%"..W.W_DIST.."s", formatNMString(dist_nm))
        local brg_aligned = string.format("%"..W.W_BRG.."s", bearing_formatted_string)

        line = line .. brg_aligned .. S.S_BRG_TO_DIST ..
               dist_aligned .. S.S_DIST_TO_ALT ..
               alt_aligned .. S.S_ALT_TO_ASPECT ..
               aspect_aligned .. cue_string
    end
    
    return line
end

-- 26. Συνάρτηση δημιουργίας επικεφαλίδων αναφοράς (ΣΤΑΘΕΡΕΣ)
local function getReportColumnHeaders(isBullseye, awacsActiveStatus)
    
    if isBullseye then
        if awacsActiveStatus then
            -- BULLSEYE AWACS ON (Ζητούμενη επικεφαλίδα)
            return "                 TYPE   BRG  / DIST       ALT         ASPECT"
        else
            -- BULLSEYE AWACS OFF (Προηγούμενη μορφή)
            return "                   TYPE    BRG / DIST         ALT        ASPECT"
        end
    else
        if awacsActiveStatus then
            -- BRAA AWACS ON (Ζητούμενη επικεφαλίδα)
            return "                TYPE    BRG   DIST        ALT         ASPECT"
        else
            -- BRAA AWACS OFF (Προηγούμενη μορφή)
            return "                   TYPE    BRG   DIST        ALT         ASPECT"
        end
    end
end


-- 27. Κύρια συνάρτηση σάρωσης
local function executeGCIScan()
    if not gci_enabled then return timer.getTime() + scan_interval_sec end 
    local player_found, player_side = locatePlayerAircraft()
    if not player_found then return timer.getTime() + 5 end    
    player_unit_ref = player_found
    local enemy_side = (player_side == coalition.side.BLUE) and coalition.side.RED or coalition.side.BLUE
    local awacs_status = isAWACSActive(player_side)
    local friendly_sensors = getSensorData(player_side)
    local enemy_air = fetchAirUnits(enemy_side)
    local enemy_rotary_wing = include_rotary_wing and fetchHeloUnits(enemy_side) or {}
    local already_seen = {}
    local reported_contacts = {}    
    local all_enemy_units = {} 
    for _, unit in ipairs(enemy_air) do table.insert(all_enemy_units, unit) end
    for _, unit in ipairs(enemy_rotary_wing) do table.insert(all_enemy_units, unit) end
    
    for _, sensor in ipairs(friendly_sensors) do
        local sensor_range_meters = fetchSensorRange(sensor) * meters_per_nautical_mile
        for _, enemy in ipairs(all_enemy_units) do
            local ok_name, name = pcall(function() return enemy:getName() end)
            if not ok_name or not name then name = "<UNKNOWN>" end            
            if not already_seen[name] and not destroyedUnits[name] and enemy.getLife and enemy:getLife() > 1 then
                local dist_meters = calculate2DDistance(sensor, enemy)
                local dist_nm_sensor = dist_meters / meters_per_nautical_mile
                local agl_feet = getAGL(enemy)                
                if dist_meters <= sensor_range_meters and agl_feet >= min_agl_feet then
                    local line_of_sight = true
                    if not isAirborneSensor(sensor) then line_of_sight = checkLOS(sensor, enemy) end                    
                    if line_of_sight then
                        already_seen[name] = true                         
                        local dist_to_player = calculate2DDistance(player_unit_ref, enemy)
                        local dist_nm_player = dist_to_player / meters_per_nautical_mile
                        local alt_unit = enemy
                        local bearing_calc = calculateBearing(player_unit_ref, enemy)
                        local aspect_calc = determineAspect(player_unit_ref, enemy)
                        local clock_cue, alt_cue = getCloseProximityCues(player_unit_ref, enemy, dist_nm_player)
                        local bullseye_brg, bullseye_dist = calculateBullseyeRef(enemy, player_side)
                        
                        local target_type_name_raw = identifyTargetType(enemy, awacs_status) -- Λήψη ονόματος με αποκοπή στους 5 χαρακτήρες
                        
                        table.insert(reported_contacts, { 
                            sortDistance = dist_to_player, 
                            line = generateReportLine(target_type_name_raw, bearing_calc, dist_nm_player, alt_unit, aspect_calc, clock_cue, alt_cue, bullseye_brg, bullseye_dist, use_bullseye_mode, awacs_status) 
                        })
                    end
                end 
            end
        end
    end    
    if #reported_contacts == 0 then 
        local no_contact_msg = string.upper("GCI Report - NO CONTACTS DETECTED") -- Κεφαλαία
        if player_unit_ref and player_unit_ref.getGroup then trigger.action.outTextForGroup(player_unit_ref:getGroup():getID(), no_contact_msg, 10, false, report_screen_position.x, report_screen_position.y) end
        return timer.getTime() + scan_interval_sec 
    end
    table.sort(reported_contacts, function(a, b) return a.sortDistance < b.sortDistance end)
    local helo_status_text = include_rotary_wing and " [+HELO]" or "" -- Κεφαλαία
    local format_type = use_bullseye_mode and "BULLSEYE" or "BRAA"
    local report_header = string.format("GCI REPORT - ENEMY CONTACTS%s\nFORMAT: %s | INTERVAL: %d SEC | MAX CONTACTS: %d\n\n", helo_status_text, format_type, scan_interval_sec, max_reported_contacts) -- Κεφαλαία
    
    local final_report = report_header
    if show_column_headers then
        final_report = final_report .. getReportColumnHeaders(use_bullseye_mode, awacs_status) .. "\n" -- Χρήση νέας συνάρτησης
    end    
    for i = 1, math.min(#reported_contacts, max_reported_contacts) do
        final_report = final_report .. reported_contacts[i].line .. "\n"
    end    
    if player_unit_ref and player_unit_ref.getGroup then
        local g = player_unit_ref:getGroup()
        if g and g.getID then
            trigger.action.outTextForGroup(g:getID(), final_report, display_duration_sec, false, report_screen_position.x, report_screen_position.y)
        end
    end    
    return timer.getTime() + scan_interval_sec
end

-- 28. Συνάρτηση Δημιουργίας Μενού F10 (Βοηθητικές συναρτήσεις)
local function getCoalitionSide(unit)
    if unit and unit.getCoalition then
        return unit:getCoalition()
    end
    return nil
end

local function resetAllSettings(groupID)
    scan_interval_sec = 15
    display_duration_sec = 15
    cue_range_nm = 5
    min_agl_feet = 3
    max_reported_contacts = 2
    gci_enabled = true
    include_rotary_wing = false
    use_bullseye_mode = false
    show_column_headers = true  
    sensorCache = {}
    destroyedUnits = {}
    sensorCacheTime = 0
    lastCleanupTime = 0
    trigger.action.outTextForGroup(groupID, "GCI SETTINGS RESET TO DEFAULTS (INTERVAL: 15S, MAX CONTACTS: 2, HEADER: ON).", 8, false, 0.5, 0.5) -- Κεφαλαία
end

local function setGCIStatus(groupID, enabled)
    gci_enabled = enabled
    trigger.action.outTextForGroup(groupID, "GCI OPERATION IS NOW " .. (enabled and "ON" or "OFF"), 5, false, 0.5, 0.5) -- Κεφαλαία
end

local function setFormatView(groupID, useBullseye)
    use_bullseye_mode = useBullseye
    local format_type = use_bullseye_mode and "BULLSEYE" or "BRAA"
    local side = getCoalitionSide(player_unit_ref)
    local bullseye = getMissionBullseye(side)
    local msg = "GCI FORMAT SET TO: " .. format_type -- Κεφαλαία
    if useBullseye and not bullseye then
        msg = "GCI FORMAT SET TO: " .. format_type .. " (BULLSEYE NOT DEFINED IN MISSION)" -- Κεφαλαία
    end
    trigger.action.outTextForGroup(groupID, msg, 5, false, 0.5, 0.5)
end

local function setHelicopters(groupID, include)
    include_rotary_wing = include
    local helo_status = include_rotary_wing and "INCLUDED" or "EXCLUDED" -- Κεφαλαία
    trigger.action.outTextForGroup(groupID, "ROTARY WING REPORTING: " .. helo_status, 5, false, 0.5, 0.5) -- Κεφαλαία
end

local function setShowHeader(groupID, show)
    show_column_headers = show
    local header_status = show_column_headers and "ON" or "OFF" -- Κεφαλαία
    trigger.action.outTextForGroup(groupID, "GCI REPORT HEADER SET TO: " .. header_status, 5, false, 0.5, 0.5) -- Κεφαλαία
end

local function setScanInterval(interval)
    scan_interval_sec = interval
    if player_unit_ref and player_unit_ref.getGroup then
        trigger.action.outTextForGroup(player_unit_ref:getGroup():getID(), "GCI SCAN INTERVAL SET TO: " .. interval .. " SECONDS.", 5, false, 0.5, 0.5) -- Κεφαλαία
    end
end

local function setMaxContacts(max)
    max_reported_contacts = max
    if player_unit_ref and player_unit_ref.getGroup then
        trigger.action.outTextForGroup(player_unit_ref:getGroup():getID(), "GCI MAX CONTACTS SET TO: " .. max .. ".", 5, false, 0.5, 0.5) -- Κεφαλαία
    end
end

local function generateFriendlyPictureReport()
    local player_found, player_side = locatePlayerAircraft()
    if not player_found then return end
    player_unit_ref = player_found
    
    local friendly_air = fetchAirUnits(player_side)
    local friendly_rotary_wing = include_rotary_wing and fetchHeloUnits(player_side) or {}
    local reported_contacts = {}
    local all_friendly_units = {}

    local awacs_status = isAWACSActive(player_side) -- AWACS status is checked
    
    for _, unit in ipairs(friendly_air) do table.insert(all_friendly_units, unit) end
    for _, unit in ipairs(friendly_rotary_wing) do table.insert(all_friendly_units, unit) end
    
    for _, unit in ipairs(all_friendly_units) do
        local ok_name, name = pcall(function() return unit:getName() end)
        if not ok_name or not name then name = "<UNKNOWN>" end -- Κεφαλαία
        
        if unit ~= player_unit_ref and not destroyedUnits[name] and unit.getLife and unit:getLife() > 1 then
            
            local dist_to_player = calculate2DDistance(player_unit_ref, unit)
            local dist_nm = dist_to_player / meters_per_nautical_mile
            local alt_unit = unit
            local bearing_calc = calculateBearing(player_unit_ref, unit)
            local aspect_calc = determineAspect(player_unit_ref, unit)
            local bullseye_brg, bullseye_dist = calculateBullseyeRef(unit, player_side)
            
            local type_name_raw = "<UNKNOWN>" -- Κεφαλαία
            if unit.getTypeName then type_name_raw = getShortTypeName(unit:getTypeName()) end -- Χρήση της νέας συνάρτησης
            
            -- ΣΗΜΑΝΤΙΚΟ: Χρησιμοποιούμε το πραγματικό AWACS status για να ενεργοποιήσουμε τη σωστή στοίχιση
            table.insert(reported_contacts, { 
                sortDistance = dist_to_player, 
                line = generateReportLine(type_name_raw, bearing_calc, dist_nm, alt_unit, aspect_calc, nil, nil, bullseye_brg, bullseye_dist, use_bullseye_mode, awacs_status) 
            })
        end
    end
    
    if #reported_contacts == 0 then 
        if player_unit_ref and player_unit_ref.getGroup then
            trigger.action.outTextForGroup(player_unit_ref:getGroup():getID(), "NO FRIENDLY AIRCRAFT DETECTED.", 5, false, report_screen_position.x, report_screen_position.y) -- Κεφαλαία
        end
        return 
    end

    table.sort(reported_contacts, function(a, b) return a.sortDistance < b.sortDistance end)
    
    local helo_status_text = include_rotary_wing and " [+HELO]" or "" -- Κεφαλαία
    local format_type = use_bullseye_mode and "BULLSEYE" or "BRAA"
    local report_header = string.format("FRIENDLY PICTURE%s\nFORMAT: %s | MAX CONTACTS: %d\n\n", helo_status_text, format_type, max_reported_contacts) -- Κεφαλαία

    
    local final_report = report_header    
    if show_column_headers then 
        final_report = final_report .. getReportColumnHeaders(use_bullseye_mode, awacs_status) .. "\n"
    end    
    for i = 1, math.min(#reported_contacts, max_reported_contacts) do
        final_report = final_report .. reported_contacts[i].line .. "\n"
    end    
    if player_unit_ref and player_unit_ref.getGroup then
        trigger.action.outTextForGroup(player_unit_ref:getGroup():getID(), final_report, display_duration_sec, false, report_screen_position.x, report_screen_position.y)
    end
end

-- 29. Συνάρτηση Δημιουργίας Μενού F10 (Αλλαγή ονομάτων)
local function createRadioMenuForGroup(group)
    local rootMenuName = "GCI"
    local groupID = group:getID()
    if menuAddedGroups[groupID] then return end

    -- 1. Root Menu
    missionCommands.addSubMenuForGroup(groupID, rootMenuName, {})
    local rootPath = {rootMenuName}

    -- 2. Υπομενού: SETTINGS 
    local settingsMenuName = "SETTINGS"
    missionCommands.addSubMenuForGroup(groupID, settingsMenuName, rootPath)
    local settingsPath = {rootMenuName, settingsMenuName}

    -- 1ο: MAX CONTACTS
    local maxContactsMenuName = "MAX CONTACTS"
    missionCommands.addSubMenuForGroup(groupID, maxContactsMenuName, settingsPath)
    local maxContactsPath = {rootMenuName, settingsMenuName, maxContactsMenuName}
    missionCommands.addCommandForGroup(groupID, "2 CONTACTS (DEFAULT)", maxContactsPath, function() setMaxContacts(2) end)
    missionCommands.addCommandForGroup(groupID, "4 CONTACTS", maxContactsPath, function() setMaxContacts(4) end)
    missionCommands.addCommandForGroup(groupID, "6 CONTACTS", maxContactsPath, function() setMaxContacts(6) end)

    -- 2ο: SCAN INTERVAL
    local intervalMenuName = "SCAN INTERVAL"
    missionCommands.addSubMenuForGroup(groupID, intervalMenuName, settingsPath)
    local intervalPath = {rootMenuName, settingsMenuName, intervalMenuName}
    missionCommands.addCommandForGroup(groupID, "15 SECONDS (DEFAULT)", intervalPath, function() setScanInterval(15) end) -- ΑΛΛΑΓΗ ΕΔΩ: Προσθήκη 60s
    missionCommands.addCommandForGroup(groupID, "30 SECONDS", intervalPath, function() setScanInterval(30) end)
    missionCommands.addCommandForGroup(groupID, "60 SECONDS", intervalPath, function() setScanInterval(60) end) -- ΑΛΛΑΓΗ ΕΔΩ: Προσθήκη 60s
    
    -- 3ο: HELICOPTERS
    local heloMenuName = "HELICOPTERS"
    missionCommands.addSubMenuForGroup(groupID, heloMenuName, settingsPath)
    local heloPath = {rootMenuName, settingsMenuName, heloMenuName}
    missionCommands.addCommandForGroup(groupID, "INCLUDE HELICOPTERS", heloPath, function() setHelicopters(groupID, true) end)
    missionCommands.addCommandForGroup(groupID, "EXCLUDE HELICOPTERS", heloPath, function() setHelicopters(groupID, false) end)

    -- 4ο: FRIENDLY PICTURE (Μετακινήθηκε και μετονομάστηκε)
    missionCommands.addCommandForGroup(groupID, "FRIENDLY PICTURE", settingsPath, function() generateFriendlyPictureReport() end)

    -- 5ο: SET BRAA 
    missionCommands.addCommandForGroup(groupID, "SET BRAA", settingsPath, function() setFormatView(groupID, false) end) 
    
    -- 6ο: SET BULLSEYE 
    missionCommands.addCommandForGroup(groupID, "SET BULLSEYE", settingsPath, function() setFormatView(groupID, true) end) 
    
    -- 7ο: HEADER VIEW 
    local headerMenuName = "HEADER VIEW"
    missionCommands.addSubMenuForGroup(groupID, headerMenuName, settingsPath)
    local headerPath = {rootMenuName, settingsMenuName, headerMenuName}
    missionCommands.addCommandForGroup(groupID, "SHOW HEADER (DEFAULT)", headerPath, function() setShowHeader(groupID, true) end)
    missionCommands.addCommandForGroup(groupID, "HIDE HEADER", headerPath, function() setShowHeader(groupID, false) end)


    -- 3. Υπομενού: CONTROL
    local statusMenuName = "CONTROL"
    missionCommands.addSubMenuForGroup(groupID, statusMenuName, rootPath)
    local statusPath = {rootMenuName, statusMenuName}
    
    -- Commands under Control
    missionCommands.addCommandForGroup(groupID, "ENABLE", statusPath, function() setGCIStatus(groupID, true) end)
    missionCommands.addCommandForGroup(groupID, "DISABLE", statusPath, function() setGCIStatus(groupID, false) end)
    missionCommands.addCommandForGroup(groupID, "CLEAR CACHE", statusPath, function()
        sensorCache = {}
        destroyedUnits = {}
        trigger.action.outTextForGroup(groupID, "GCI CACHES CLEARED.", 5, false, 0.5, 0.5) -- Κεφαλαία
    end)  
    missionCommands.addCommandForGroup(groupID, "RESET ALL", statusPath, function() resetAllSettings(groupID) end)

    menuAddedGroups[groupID] = true
end


-- 30. Συνάρτηση για συνεχή έλεγχο και προσθήκη του μενού αν είναι έτοιμο
local function continuousMenuInit()
    local allGroups = {}
    -- Συλλογή όλων των ομάδων
    for _, side in ipairs({coalition.side.BLUE, coalition.side.RED}) do
        for _, group in ipairs(coalition.getGroups(side)) do
            table.insert(allGroups, group)
        end
    end

    -- Έλεγχος αν η ομάδα είναι client και αν χρειάζεται προσθήκη μενού
    for _, group in ipairs(allGroups) do
        local isClientGroup = false
        for _, unit in ipairs(group:getUnits()) do
            -- Ελέγχουμε αν υπάρχει ενεργός παίκτης στη μονάδα
            if unit and unit.getPlayerName and unit:getPlayerName() then
                isClientGroup = true
                break
            end
        end

        if isClientGroup and not menuAddedGroups[group:getID()] then
            createRadioMenuForGroup(group)
        end
    end
    return timer.getTime() + 10
end


-- 31. Συνάρτηση καθαρισμού κατεστραμμένων μονάδων
local function cleanupDeadUnits()
    local current_time = timer.getTime()
    if current_time - lastCleanupTime > dead_unit_cleanup_sec then
        destroyedUnits = {}
        lastCleanupTime = current_time
    end
    return current_time + dead_unit_cleanup_sec
end

-- 32. Συνάρτηση που καλεί την κύρια λογική σάρωσης
local function mainGCIScan()
    return executeGCIScan()
end

-- VI. ΕΚΚΙΝΗΣΗ
trigger.action.outText("GCI V.5.06 INITIALIZING...", 7) -- Κεφαλαία
timer.scheduleFunction(cleanupDeadUnits, nil, timer.getTime() + dead_unit_cleanup_sec)
timer.scheduleFunction(continuousMenuInit, nil, 1) 
timer.scheduleFunction(mainGCIScan, nil, 10)